<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\MassDestroyIdeaRequest;
use App\Http\Requests\StoreIdeaRequest;
use App\Http\Requests\UpdateIdeaRequest;
use App\Models\Idea;
use App\Models\Producttype;
use Gate;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Yajra\DataTables\Facades\DataTables;

class IdeasController extends Controller
{
    public function index(Request $request)
    {
        abort_if(Gate::denies('idea_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        if ($request->ajax()) {
            $query = Idea::with(['productline'])->select(sprintf('%s.*', (new Idea)->table));
            $table = Datatables::of($query);

            $table->addColumn('placeholder', '&nbsp;');
            $table->addColumn('actions', '&nbsp;');

            $table->editColumn('actions', function ($row) {
                $viewGate      = 'idea_show';
                $editGate      = 'idea_edit';
                $deleteGate    = 'idea_delete';
                $crudRoutePart = 'ideas';

                return view('partials.datatablesActions', compact(
                    'viewGate',
                    'editGate',
                    'deleteGate',
                    'crudRoutePart',
                    'row'
                ));
            });

            $table->editColumn('id', function ($row) {
                return $row->id ? $row->id : '';
            });
            $table->editColumn('name', function ($row) {
                return $row->name ? $row->name : '';
            });
            $table->addColumn('productline_name', function ($row) {
                return $row->productline ? $row->productline->name : '';
            });

            $table->editColumn('link', function ($row) {
                return $row->link ? $row->link : '';
            });
            $table->editColumn('added', function ($row) {
                return $row->added ? Idea::ADDED_SELECT[$row->added] : '';
            });

            $table->rawColumns(['actions', 'placeholder', 'productline']);

            return $table->make(true);
        }

        return view('admin.ideas.index');
    }

    public function create()
    {
        abort_if(Gate::denies('idea_create'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $productlines = Producttype::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        return view('admin.ideas.create', compact('productlines'));
    }

    public function store(StoreIdeaRequest $request)
    {
        $idea = Idea::create($request->all());

        return redirect()->route('admin.ideas.index');
    }

    public function edit(Idea $idea)
    {
        abort_if(Gate::denies('idea_edit'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $productlines = Producttype::pluck('name', 'id')->prepend(trans('global.pleaseSelect'), '');

        $idea->load('productline');

        return view('admin.ideas.edit', compact('idea', 'productlines'));
    }

    public function update(UpdateIdeaRequest $request, Idea $idea)
    {
        $idea->update($request->all());

        return redirect()->route('admin.ideas.index');
    }

    public function show(Idea $idea)
    {
        abort_if(Gate::denies('idea_show'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $idea->load('productline');

        return view('admin.ideas.show', compact('idea'));
    }

    public function destroy(Idea $idea)
    {
        abort_if(Gate::denies('idea_delete'), Response::HTTP_FORBIDDEN, '403 Forbidden');

        $idea->delete();

        return back();
    }

    public function massDestroy(MassDestroyIdeaRequest $request)
    {
        $ideas = Idea::find(request('ids'));

        foreach ($ideas as $idea) {
            $idea->delete();
        }

        return response(null, Response::HTTP_NO_CONTENT);
    }
}
